#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Thu Oct 12 14:42:49 2023

@author: badler

#convert lidar hpl files to netcdf 
"""
import os
import numpy as np
import matplotlib.dates as mdates 
import matplotlib.pyplot as plt
import glob
import xarray as xr
import datetime
import pandas as pd

def num2datestr(*args):
        x = args[0]
        if len(args)==1:
                str_format='%Y%m%d%H%M%S'
        else:
                str_format=args[1]
        d=mdates.num2date(x)
        #s=[]
        #for i in range(0,len(d)):
        #       line = d[i]
        #       x=line.strftime(str_format)
        #       s.append(x)
        s=d.strftime(str_format)

        return s

def datestr2num(*args):
    x=args[0]
    if len(args)==1:
        str_format='%Y%m%d'
    else:
        str_format=args[1]
    n=mdates.date2num(datetime.datetime.strptime(x,str_format))
    return n


datestring=os.getenv('pdate')
siteid=os.getenv('siteid')
site=os.getenv('site')
campaigndir=os.getenv('campaigndir')
rawlidardir=os.getenv('rawlidardir')
lidarfid=os.getenv('lidarfid')
lidarheading=float(os.getenv('lidarheading'))
#sitelat=os.getenv('sitelat')
#sitelon=os.getenv('sitelon')
#sitealt=os.getenv('sitealt')
lidar=os.getenv('lidar')
lidarraw=os.getenv('lidarraw')

if not os.path.exists(os.path.join(campaigndir,siteid,lidar,'netcdf')):
    os.makedirs(os.path.join(campaigndir,siteid,lidar,'netcdf'))

datestring_=datestring[:4]+'-'+datestring[4:6]+'-'+datestring[6:]

#Stare
listdir = sorted(glob.glob(os.path.join(rawlidardir,datestring,'Stare_'+lidarraw[4:]+'_'+datestring+'*.hpl')))
if len(listdir) > 0:
    fvar=['time','azimuth','elevation','pitch','roll','velocity','intensity','backscatter','spectral_width']
    IN={}
    for fv in fvar:
        IN[fv]=[]
        
    # for l in listdir[:1]:
    fcount=0
    for l in listdir:
        print('read '+l)
        with  open(l) as f:
            #header
            INh={}
            for line in f:
                if '****' in line :
                    break
                if not ':' in line:
                    continue
                dummy=line.split(':')
                INh[dummy[0]]=dummy[1].strip()  
          
            rgcount=-1
            # int(INh['Number of gates']))
            if fcount ==0:
                for fv in fvar[5:]:
                    IN[fv]=[[] for j in range(int(INh['Number of gates']))]
                fcount=1
            for line in f:
                dummy=line.split()
                if len(dummy)<=3: #incomplete line
                    continue
                if rgcount==-1:
                    IN['time'].append(float(dummy[0]))
                    IN['azimuth'].append(float(dummy[1]))
                    IN['elevation'].append(float(dummy[2]))
                    IN['pitch'].append(float(dummy[3]))
                    IN['roll'].append(float(dummy[4]))
                    
                    rgcount=0
                else:
                    try:
                        rgcount=int(dummy[0])
                    except: #if this fails, it is likely because range gate number changed
                        print('range gate number changed')
                    IN['velocity'][rgcount].append(float(dummy[1]))
                    IN['intensity'][rgcount].append(float(dummy[2]))
                    IN['backscatter'][rgcount].append(float(dummy[3]))
                    if len(dummy)==5:
                        IN['spectral_width'][rgcount].append(float(dummy[4]))
                    else:
                            IN['spectral_width'][rgcount].append(float(-999))
                if rgcount==int(INh['Number of gates'])-1:
                    rgcount=-1
    for fv in fvar:
        IN[fv]=np.asarray(IN[fv])
                 
    rglength=float(INh['Range gate length (m)'])
    rgcount=int(INh['Number of gates'])
    IN['range']=(np.arange(0,rgcount)+0.5)*rglength/1000
    IN['datetime']=datestr2num(INh['Start time'].split()[0])+IN['time']/24

    # add lidarheading to azimuth angle and make sure that it is less than 360 deg
    IN['azimuth'] = (IN['azimuth']+lidarheading) % 360

    #transpose
    for fv in fvar[5:]:
        IN[fv]=IN[fv].T
        
    #save as netcdf
    dt=mdates.num2date(IN['datetime'])
    dt=[d.replace(tzinfo=None) for d in dt]
    ds=xr.Dataset(
        data_vars=dict(
            azimuth=('time',IN['azimuth']),
            elevation=('time',IN['elevation']),
            pitch=('time',IN['pitch']),
            roll=('time',IN['roll']),
            velocity=(['time','range'],IN['velocity']),
            intensity=(['time','range'],IN['intensity']),
            backscatter=(['time','range'],IN['backscatter']),
            spectral_width=(['time','range'],IN['spectral_width']),
            ),
        coords=dict(
            time=('time',dt),
            range=('range',IN['range']),
            ),
        attrs=dict(
            comment1='Hourly Halo hpl files are converted to daily netcdf files',
            comment2='converted by bianca.adler@noaa.gov at '+str(datetime.datetime.now(datetime.UTC))+' UTC',
            inputfiles=listdir,
            header_last_file=str(INh).strip()),
        )
    # add information on lidar heading
    ds['heading'] = lidarheading
    ds['heading'].attrs={'units':'deg','long_name':'Lidar heading that was added to the azimuth angles'}
    ds['azimuth'].attrs={'units':'deg','long_name':'Azimuth angle'}
    ds['elevation'].attrs={'units':'deg','long_name':'Elevation angle (90 is zenith)'}
    ds['range'].attrs={'units':'km','long_name':'Range gate'}
    ds['roll'].attrs={'units':'deg'}
    ds['pitch'].attrs={'units':'deg'}
    ds['velocity'].attrs={'units':'m/s','long_name':'Radial velocity'}
    ds['intensity'].attrs={'units':'Intensity (SNR+1)','long_name':'Intensity'}
    ds['backscatter'].attrs={'units':'Beta (m-1 sr-1)','long_name':'Backscatter'}
    ds['spectral_width'].attrs={'units':'m/s','long_name':'Spectral width'}

    ds['base_time']=int(pd.to_datetime(datestring).timestamp())
    ds['time_offset'] = ('time',[pd.to_datetime(d_).timestamp()-int(ds['base_time']) for d_ in ds.time.values])

    #make sure that time is sorted
    ds=ds.sortby(ds.time)
    elestr=str(int(np.median(np.round(ds.elevation))))
    print('save to netcdf')
    fout=os.path.join(campaigndir,siteid,lidar,'netcdf',siteid+'.'+lidarraw+'.a1.stare'+elestr+'.'+datestring+'.cdf')
    if os.path.isfile(fout):
        os.remove(fout)
    ds.to_netcdf(fout)
    #set right permission
    os.chmod(fout,0o664)        


#User
listdir = sorted(glob.glob(os.path.join(rawlidardir,'data',datestring,lidarfid+'_'+lidarraw[4:]+'_'+datestring+'*.hpl')))
fvar=['time','azimuth','elevation','pitch','roll','velocity','intensity','backscatter','spectral_width']
IN={}
for fv in fvar:
    IN[fv]=[]

# do not read last file with scans since this might be incomplete
listdir = listdir[:-1]
    
# for l in listdir[:1]:
fcount=0
for l in listdir:
    print('read '+l)
    with  open(l) as f:
        #header
        INh={}
        for line in f:
            if '****' in line :
                break
            if not ':' in line:
                continue
            dummy=line.split(':')
            INh[dummy[0]]=dummy[1].strip()  
      
        rgcount=-1
        # int(INh['Number of gates']))
        if fcount ==0:
            for fv in fvar[5:]:
                IN[fv]=[[] for j in range(int(INh['Number of gates']))]
            fcount=1
        for line in f:
            dummy=line.split()
            if len(dummy)<=3: #incomplete line
                continue
            if rgcount==-1:
                IN['time'].append(float(dummy[0]))
                IN['azimuth'].append(float(dummy[1]))
                IN['elevation'].append(float(dummy[2]))
                IN['pitch'].append(float(dummy[3]))
                IN['roll'].append(float(dummy[4]))
                
                rgcount=0
            else:
                try:
                    rgcount=int(dummy[0])
                except: #if this fails, it is likely because range gate number changed
                    print('range gate number changed')
                IN['velocity'][rgcount].append(float(dummy[1]))
                IN['intensity'][rgcount].append(float(dummy[2]))
                IN['backscatter'][rgcount].append(float(dummy[3]))
                if len(dummy)==5:
            	    IN['spectral_width'][rgcount].append(float(dummy[4]))
                else:
	                IN['spectral_width'][rgcount].append(float(-999))
            if rgcount==int(INh['Number of gates'])-1:
                rgcount=-1
for fv in fvar:
    IN[fv]=np.asarray(IN[fv])
rglength=float(INh['Range gate length (m)'])
rgcount=int(INh['Number of gates'])
IN['range']=(np.arange(0,rgcount)+0.5)*rglength/1000
IN['datetime']=datestr2num(INh['Start time'].split()[0])+IN['time']/24

# add lidarheading to azimuth angle and make sure that it is less than 360 deg
IN['azimuth'] = (IN['azimuth']+lidarheading) % 360

#transpose
for fv in fvar[5:]:
    IN[fv]=IN[fv].T
    
#save as netcdf
dt=mdates.num2date(IN['datetime'])
dt=[d.replace(tzinfo=None) for d in dt]
ds=xr.Dataset(
    data_vars=dict(
        azimuth=('time',IN['azimuth']),
        elevation=('time',IN['elevation']),
        pitch=('time',IN['pitch']),
        roll=('time',IN['roll']),
        velocity=(['time','range'],IN['velocity']),
        intensity=(['time','range'],IN['intensity']),
        backscatter=(['time','range'],IN['backscatter']),
        spectral_width=(['time','range'],IN['spectral_width']),
        ),
    coords=dict(
        time=('time',dt),
        range=('range',IN['range']),
        ),
    attrs=dict(
        comment1='Hourly Halo hpl files are converetd to daily netcdf files',
        comment2='converted by bianca.adler@noaa.gov at '+str(datetime.datetime.now(datetime.UTC))+' UTC',
        inputfiles=listdir,
        header_last_file=str(INh).strip()),
    )
ds['heading'] = lidarheading
ds['heading'].attrs={'units':'deg','long_name':'Lidar heading that was added to the azimuth angles'}
ds['azimuth'].attrs={'units':'deg','long_name':'Azimuth angle'}
ds['elevation'].attrs={'units':'deg','long_name':'Elevation angle (90 is zenith)'}
ds['range'].attrs={'units':'km','long_name':'Range gate'}
ds['roll'].attrs={'units':'deg'}
ds['pitch'].attrs={'units':'deg'}
ds['velocity'].attrs={'units':'m/s','long_name':'Radial velocity'}
ds['intensity'].attrs={'units':'Intensity (SNR+1)','long_name':'Intensity'}
ds['backscatter'].attrs={'units':'Beta (m-1 sr-1)','long_name':'Backscatter'}
ds['spectral_width'].attrs={'units':'m/s','long_name':'Spectral width'}

ds['base_time']=int(pd.to_datetime(datestring).timestamp())
ds['time_offset'] = ('time',[pd.to_datetime(d_).timestamp()-int(ds['base_time']) for d_ in ds.time.values])

#make sure that time is sorted
ds=ds.sortby(ds.time)
fout=os.path.join(campaigndir,siteid,lidar,'netcdf',siteid+'.'+lidarraw+'.a1.'+lidarfid+'.'+datestring+'.cdf')
if os.path.isfile(fout):
    os.remove(fout)
print('save to netcdf')
ds.to_netcdf(fout)
#set right permission
os.chmod(fout,0o664)        

# because WINDoe currently cannot handle file with 2 different elevation angles I separate both scans and store them in different directories
# find unique elevation angles (rounded accuracy only)
el = ds.elevation.values
azu = np.unique(np.round(ds.azimuth.values))
elu = np.unique(np.round(ds.elevation.values))
felu = []
fel = []
for el_ in elu:
    idx = np.where(np.round(el) == el_)[0]
    felu.append(len(idx))
    fel.append(idx)
#sort in ascending order
idx = np.argsort(felu)
felus = np.array(felu)[idx]
elus = np.array(elu)[idx]
# pick two most frequent elevation angle (assumes that two different elevation angles were scanned!!!)

for el in elus[-2:]:
    idx = np.where(np.round(ds.elevation.values) == el)[0]
    ds_ = ds.isel(time=idx)
    fout=os.path.join(campaigndir,siteid,lidar,'netcdf','elevation'+str(int(el)),siteid+'.'+lidarraw+'.a1.'+lidarfid+'.elevation'+str(int(el))+'.'+datestring+'.cdf')
    if not os.path.exists(os.path.join(campaigndir,siteid,lidar,'netcdf','elevation'+str(int(el)))):
        os.makedirs(os.path.join(campaigndir,siteid,lidar,'netcdf','elevation'+str(int(el))))
    print('save netcdf file for elevation angle '+str(int(el))+' to '+fout)
    
    ds_.to_netcdf(fout)
    #set right permission
    os.chmod(fout,0o664)        



